@extends('admin.layouts.app')

@section('title', 'Media Library')

@section('page-header')
<div class="flex items-center justify-between">
    <div>
        <h1 class="text-3xl font-bold text-gray-900 dark:text-white">Media Library</h1>
        <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Manage your uploaded images, videos, and files</p>
    </div>
    <div class="flex space-x-3">
        <button onclick="document.getElementById('media-upload').click()" 
                class="inline-flex items-center px-4 py-2 bg-cyan-600 hover:bg-cyan-700 text-white font-medium rounded-lg transition-colors">
            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4" />
            </svg>
            Upload Media
        </button>
        <input type="file" id="media-upload" multiple accept="image/*,video/*" class="hidden" onchange="handleMediaUpload(event)">
    </div>
</div>
@endsection

@section('content')
{{-- Media Grid --}}
<div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
    {{-- Filter Tabs --}}
    <div class="flex space-x-4 mb-6 border-b border-gray-200 dark:border-gray-700">
        <button class="px-4 py-2 text-sm font-medium text-cyan-600 dark:text-cyan-400 border-b-2 border-cyan-600 dark:border-cyan-400">
            All Media
        </button>
        <button class="px-4 py-2 text-sm font-medium text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300">
            Images
        </button>
        <button class="px-4 py-2 text-sm font-medium text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300">
            Videos
        </button>
        <button class="px-4 py-2 text-sm font-medium text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300">
            Documents
        </button>
    </div>
    
    {{-- Media Grid --}}
    <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4" id="media-grid">
        @forelse($media as $item)
            <div class="group relative bg-gray-100 dark:bg-gray-700 rounded-lg overflow-hidden aspect-square">
                @if($item->isImage())
                    <img src="{{ Storage::url($item->path) }}" 
                         alt="{{ $item->alt_text ?? $item->name }}"
                         class="w-full h-full object-cover">
                @else
                    <div class="w-full h-full flex items-center justify-center">
                        <svg class="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                    </div>
                @endif
                <div class="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-50 transition-opacity flex items-center justify-center opacity-0 group-hover:opacity-100">
                    <div class="flex space-x-2">
                        <button onclick="selectMedia('{{ Storage::url($item->path) }}')" 
                                class="px-3 py-1.5 bg-cyan-600 hover:bg-cyan-700 text-white text-sm rounded">
                            Select
                        </button>
                        <form action="{{ route('admin.media.destroy', $item) }}" method="POST" class="inline" onsubmit="return confirm('Delete this media?');">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="px-3 py-1.5 bg-red-600 hover:bg-red-700 text-white text-sm rounded">
                                Delete
                            </button>
                        </form>
                    </div>
                </div>
                <div class="absolute bottom-0 left-0 right-0 bg-black bg-opacity-75 text-white text-xs p-2 truncate">
                    {{ $item->name }}
                </div>
            </div>
        @empty
            <div class="col-span-full text-center py-12 text-gray-500 dark:text-gray-400">
                <svg class="w-16 h-16 mx-auto mb-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                </svg>
                <p class="text-lg font-medium mb-2">No media files yet</p>
                <p class="text-sm">Upload your first image or file to get started</p>
            </div>
        @endforelse
    </div>
    
    @if($media->hasPages())
        <div class="mt-6">
            {{ $media->links() }}
        </div>
    @endif
</div>

{{-- Media Browser Modal (for rich editor) --}}
<div id="media-browser-modal" 
     x-data="{ open: false }"
     x-show="open"
     @open-media-browser.window="open = true"
     @close.window="open = false"
     class="fixed inset-0 z-50 hidden"
     style="display: none;"
     x-cloak>
    <div class="fixed inset-0 bg-black bg-opacity-50" @click="open = false"></div>
    <div class="fixed inset-4 bg-white dark:bg-gray-800 rounded-lg shadow-xl flex flex-col">
        <div class="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
            <h2 class="text-2xl font-bold text-gray-900 dark:text-white">Select Media</h2>
            <button @click="open = false" class="text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                </svg>
            </button>
        </div>
        <div class="flex-1 overflow-auto p-6">
            <div class="grid grid-cols-4 gap-4" id="browser-media-grid">
                {{-- Media items for browser --}}
            </div>
        </div>
        <div class="p-6 border-t border-gray-200 dark:border-gray-700 flex justify-end space-x-3">
            <button @click="open = false" class="px-4 py-2 bg-gray-600 hover:bg-gray-700 text-white rounded-lg">
                Cancel
            </button>
            <button @click="insertSelectedMedia()" class="px-4 py-2 bg-cyan-600 hover:bg-cyan-700 text-white rounded-lg">
                Insert Selected
            </button>
        </div>
    </div>
</div>

@push('scripts')
<script>
function handleMediaUpload(event) {
    const files = event.target.files;
    if (!files.length) return;
    
    const formData = new FormData();
    Array.from(files).forEach(file => {
        formData.append('files[]', file);
    });
    
    fetch('{{ route("admin.media.upload") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
        },
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        }
    })
    .catch(error => {
        console.error('Upload error:', error);
        alert('Upload failed. Please try again.');
    });
}

function selectMedia(url) {
    window.dispatchEvent(new CustomEvent('media-selected', { detail: { url } }));
    document.getElementById('media-browser-modal').querySelector('[x-data]').__x.$data.open = false;
}

window.addEventListener('media-selected', function(e) {
    // Insert into active editor if available
    const activeEditor = window.activeEditor;
    if (activeEditor) {
        const img = document.createElement('img');
        img.src = e.detail.url;
        img.className = 'max-w-full h-auto';
        activeEditor.value += '\n<img src="' + e.detail.url + '" alt="">\n';
    }
});
</script>
@endpush
@endsection

