// Tiptap Rich Text Editor for Admin Panel
import { Editor } from '@tiptap/core';
import StarterKit from '@tiptap/starter-kit';
import Image from '@tiptap/extension-image';
import Link from '@tiptap/extension-link';
import Placeholder from '@tiptap/extension-placeholder';

// Initialize Tiptap Editor
export function initTiptapEditor(elementId, content = '') {
    const editorElement = document.getElementById(elementId);
    if (!editorElement) return null;

    const editor = new Editor({
        element: editorElement,
        extensions: [
            StarterKit.configure({
                heading: {
                    levels: [1, 2, 3, 4, 5, 6],
                },
            }),
            Image.configure({
                inline: true,
                allowBase64: true,
            }),
            Link.configure({
                openOnClick: false,
                HTMLAttributes: {
                    class: 'text-cyan-600 dark:text-cyan-400 hover:underline',
                },
            }),
            Placeholder.configure({
                placeholder: 'Start writing your content...',
            }),
        ],
        content: content,
        editorProps: {
            attributes: {
                class: 'prose prose-lg dark:prose-invert max-w-none focus:outline-none min-h-[400px] p-4',
            },
        },
    });

    return editor;
}

// Toolbar Component
export function createToolbar(editor, containerId) {
    const container = document.getElementById(containerId);
    if (!container || !editor) return;

    const buttons = [
        { name: 'bold', icon: 'B', command: () => editor.chain().focus().toggleBold().run() },
        { name: 'italic', icon: 'I', command: () => editor.chain().focus().toggleItalic().run() },
        { name: 'heading1', icon: 'H1', command: () => editor.chain().focus().toggleHeading({ level: 1 }).run() },
        { name: 'heading2', icon: 'H2', command: () => editor.chain().focus().toggleHeading({ level: 2 }).run() },
        { name: 'heading3', icon: 'H3', command: () => editor.chain().focus().toggleHeading({ level: 3 }).run() },
        { name: 'bulletList', icon: '•', command: () => editor.chain().focus().toggleBulletList().run() },
        { name: 'orderedList', icon: '1.', command: () => editor.chain().focus().toggleOrderedList().run() },
        { name: 'blockquote', icon: '"', command: () => editor.chain().focus().toggleBlockquote().run() },
        { name: 'code', icon: '</>', command: () => editor.chain().focus().toggleCode().run() },
        { name: 'link', icon: '🔗', command: () => {
            const url = window.prompt('Enter URL:');
            if (url) editor.chain().focus().setLink({ href: url }).run();
        }},
        { name: 'image', icon: '🖼️', command: () => {
            // Open media browser
            window.dispatchEvent(new CustomEvent('open-media-browser', { detail: { editor } }));
        }},
    ];

    buttons.forEach(btn => {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'px-3 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors';
        button.textContent = btn.icon;
        button.title = btn.name;
        button.onclick = btn.command;
        container.appendChild(button);
    });
}

