<?php

namespace App\Services\Parsers;

use Illuminate\Support\Str;

class SeederParser implements ParserInterface
{
    public function parse(string $content): array
    {
        // Extract data from Laravel seeder-like syntax
        $title = '';
        $summary = '';
        $contentHtml = '';
        $pros = [];
        $cons = [];
        $features = [];
        $faqs = [];
        $affiliateLink = '';
        $cta = '';

        // Extract title
        if (preg_match('/["\']title["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $title = $matches[1];
        } elseif (preg_match('/["\']name["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $title = $matches[1];
        }

        // Extract summary/description
        if (preg_match('/["\'](?:summary|excerpt|description)["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $summary = $matches[1];
        }

        // Extract content
        if (preg_match('/["\'](?:content|body)["\']\s*=>\s*["\']([^"\']+)["\']/s', $content, $matches)) {
            $contentHtml = $matches[1];
        }

        // Extract pros array
        if (preg_match('/["\']pros["\']\s*=>\s*\[(.*?)\]/s', $content, $matches)) {
            $pros = $this->extractArrayItems($matches[1]);
        }

        // Extract cons array
        if (preg_match('/["\']cons["\']\s*=>\s*\[(.*?)\]/s', $content, $matches)) {
            $cons = $this->extractArrayItems($matches[1]);
        }

        // Extract features array
        if (preg_match('/["\']features["\']\s*=>\s*\[(.*?)\]/s', $content, $matches)) {
            $features = $this->extractArrayItems($matches[1]);
        }

        // Extract FAQs array
        if (preg_match('/["\']faqs["\']\s*=>\s*\[(.*?)\]/s', $content, $matches)) {
            $faqs = $this->extractFAQsFromArray($matches[1]);
        }

        // Extract affiliate link
        if (preg_match('/["\'](?:affiliate_link|affiliate_url|link)["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $affiliateLink = $matches[1];
        }

        // Extract CTA
        if (preg_match('/["\']cta["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $cta = $matches[1];
        }

        // Extract meta title
        $metaTitle = $title;
        if (preg_match('/["\']meta_title["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $metaTitle = $matches[1];
        }

        // Extract meta description
        $metaDescription = $summary;
        if (preg_match('/["\']meta_description["\']\s*=>\s*["\']([^"\']+)["\']/', $content, $matches)) {
            $metaDescription = $matches[1];
        }

        // Extract keywords
        $keywords = [];
        if (preg_match('/["\']keywords["\']\s*=>\s*\[(.*?)\]/s', $content, $matches)) {
            $keywords = $this->extractArrayItems($matches[1]);
        }

        return [
            'title' => $title,
            'slug' => $title ? Str::slug($title) : '',
            'summary' => $summary,
            'content_html' => $contentHtml,
            'content_markdown' => '',
            'affiliate_link' => $affiliateLink,
            'pros' => $pros,
            'cons' => $cons,
            'features' => $features,
            'faqs' => $faqs,
            'tables' => [],
            'cta' => $cta,
            'seo' => [
                'meta_title' => $metaTitle,
                'meta_description' => $metaDescription,
                'keywords' => $keywords,
            ],
        ];
    }

    protected function extractArrayItems(string $arrayContent): array
    {
        $items = [];
        
        // Match quoted strings
        preg_match_all('/["\']([^"\']+)["\']/', $arrayContent, $matches);
        
        if (!empty($matches[1])) {
            $items = $matches[1];
        }

        return $items;
    }

    protected function extractFAQsFromArray(string $arrayContent): array
    {
        $faqs = [];

        // Match associative array items like 'question' => 'answer'
        if (preg_match_all('/\[(.*?)\]/s', $arrayContent, $matches)) {
            foreach ($matches[1] as $faqItem) {
                $question = '';
                $answer = '';

                if (preg_match('/["\'](?:question|q)["\']\s*=>\s*["\']([^"\']+)["\']/', $faqItem, $qMatches)) {
                    $question = $qMatches[1];
                }

                if (preg_match('/["\'](?:answer|a)["\']\s*=>\s*["\']([^"\']+)["\']/', $faqItem, $aMatches)) {
                    $answer = $aMatches[1];
                }

                if (!empty($question) || !empty($answer)) {
                    $faqs[] = [
                        'question' => $question,
                        'answer' => $answer,
                    ];
                }
            }
        }

        return $faqs;
    }
}

