<?php

namespace App\Services\Parsers;

use Illuminate\Support\Str;

class JSONParser implements ParserInterface
{
    public function parse(string $content): array
    {
        $data = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \InvalidArgumentException('Invalid JSON: ' . json_last_error_msg());
        }

        // Handle array of items (take first item)
        if (isset($data[0]) && is_array($data[0])) {
            $data = $data[0];
        }

        return $this->normalize($data);
    }

    protected function normalize(array $data): array
    {
        // Build content_html from introduction or existing content
        $contentHtml = $data['content_html'] ?? $data['content'] ?? $data['body'] ?? '';
        if (empty($contentHtml) && !empty($data['introduction'])) {
            $contentHtml = '<p>' . htmlspecialchars($data['introduction'], ENT_QUOTES, 'UTF-8') . '</p>';
        } elseif (!empty($data['introduction']) && empty($contentHtml)) {
            $contentHtml = '<p>' . htmlspecialchars($data['introduction'], ENT_QUOTES, 'UTF-8') . '</p>';
        }
        
        return [
            'title' => $data['title'] ?? $data['name'] ?? $data['heading'] ?? '',
            'slug' => $data['slug'] ?? (isset($data['title']) ? Str::slug($data['title']) : ''),
            'summary' => $data['summary'] ?? $data['excerpt'] ?? $data['description'] ?? $data['short_description'] ?? '',
            'content_html' => $contentHtml,
            'content_markdown' => $data['content_markdown'] ?? $data['markdown'] ?? '',
            'affiliate_link' => $data['affiliate_link'] ?? $data['affiliate_url'] ?? $data['link'] ?? '',
            'pros' => $this->normalizeArray($data['pros'] ?? $data['advantages'] ?? []),
            'cons' => $this->normalizeArray($data['cons'] ?? $data['disadvantages'] ?? []),
            'features' => $this->normalizeArray($data['features'] ?? $data['feature_list'] ?? []),
            'faqs' => $this->normalizeFAQs($data['faqs'] ?? $data['frequently_asked_questions'] ?? []),
            'tables' => $this->normalizeArray($data['tables'] ?? []),
            'cta' => $data['cta'] ?? $data['cta_section'] ?? $data['call_to_action'] ?? '',
            'cta_section' => $data['cta_section'] ?? $data['cta'] ?? $data['call_to_action'] ?? '',
            'seo' => [
                'meta_title' => $data['seo']['meta_title'] ?? $data['meta_title'] ?? $data['title'] ?? '',
                'meta_description' => $data['seo']['meta_description'] ?? $data['meta_description'] ?? $data['summary'] ?? '',
                'keywords' => $this->normalizeArray($data['seo']['keywords'] ?? $data['keywords'] ?? []),
            ],
        ];
    }

    protected function normalizeArray($value): array
    {
        if (is_string($value)) {
            return array_filter(array_map('trim', explode(',', $value)));
        }
        if (is_array($value)) {
            $result = [];
            foreach ($value as $item) {
                // If item is an array, try to extract string value or convert to string
                if (is_array($item)) {
                    // Try to get 'name', 'title', or first element
                    $stringValue = $item['name'] ?? $item['title'] ?? $item[0] ?? null;
                    if ($stringValue !== null) {
                        $result[] = (string)$stringValue;
                    } elseif (!empty($item)) {
                        // If array has content but no obvious string key, convert to JSON
                        $result[] = json_encode($item);
                    }
                } else {
                    // Simple string value
                    $result[] = (string)$item;
                }
            }
            return array_values(array_filter($result));
        }
        return [];
    }

    protected function normalizeFAQs($value): array
    {
        if (!is_array($value)) {
            return [];
        }

        $faqs = [];
        foreach ($value as $faq) {
            if (is_array($faq)) {
                $faqs[] = [
                    'question' => $faq['question'] ?? $faq['q'] ?? '',
                    'answer' => $faq['answer'] ?? $faq['a'] ?? '',
                ];
            } elseif (is_string($faq)) {
                // Try to split by common separators
                if (strpos($faq, ':') !== false) {
                    [$q, $a] = explode(':', $faq, 2);
                    $faqs[] = [
                        'question' => trim($q),
                        'answer' => trim($a),
                    ];
                } else {
                    $faqs[] = [
                        'question' => '',
                        'answer' => $faq,
                    ];
                }
            }
        }
        return $faqs;
    }
}

