<?php

namespace App\Services;

use App\Services\Parsers\JSONParser;
use App\Services\Parsers\MarkdownParser;
use App\Services\Parsers\HTMLParser;
use App\Services\Parsers\SeederParser;
use App\Services\Parsers\TextParser;

class ContentParserService
{
    protected $parsers = [];

    public function __construct()
    {
        $this->parsers = [
            'json' => new JSONParser(),
            'markdown' => new MarkdownParser(),
            'html' => new HTMLParser(),
            'seeder' => new SeederParser(),
            'text' => new TextParser(),
        ];
    }

    /**
     * Auto-detect content format
     */
    public function detectFormat(string $content): string
    {
        $content = trim($content);

        // JSON detection - starts with { or [
        if (preg_match('/^\s*[{\[]/', $content)) {
            return 'json';
        }

        // Markdown detection - contains markdown patterns
        if (preg_match('/^#{1,6}\s+/m', $content) || 
            preg_match('/^\s*[-*+]\s+/m', $content) ||
            preg_match('/^\s*\d+\.\s+/m', $content) ||
            preg_match('/\*\*.*\*\*|__.*__/', $content)) {
            return 'markdown';
        }

        // HTML detection - contains HTML tags
        if (preg_match('/<[a-z][\s\S]*>/i', $content)) {
            return 'html';
        }

        // Seeder detection - contains Laravel seeder patterns
        if (preg_match('/::create\s*\(|=>\s*\[|DB::table/', $content)) {
            return 'seeder';
        }

        // Default to text
        return 'text';
    }

    /**
     * Parse content using specified format
     */
    public function parse(string $content, string $format): array
    {
        if (!isset($this->parsers[$format])) {
            throw new \InvalidArgumentException("Unsupported format: {$format}");
        }

        $parser = $this->parsers[$format];
        return $parser->parse($content);
    }

    /**
     * Get available parsers
     */
    public function getAvailableParsers(): array
    {
        return array_keys($this->parsers);
    }
}

